const mongoose = require('mongoose');

const BookingSchema = new mongoose.Schema({
    user_id: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'User', // Mengacu pada model User jika diperlukan
        required: true
    },
    borrower: {
        type: String,
        required: true
    },
    borrowDate: {
        type: Date,
        required: true
    },
    returnDate: {
        type: Date,
        required: true
    },
    returned: {
        type: Boolean,
        default: false
    },
    notes: {
        type: String,
        default: ''
    },
    workshop: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'Workshop', // Mengacu pada model Workshop jika diperlukan
        required: true
    }
}, { timestamps: true });

const WorkshopSchema = new mongoose.Schema({
    name: {
        type: String,
        unique: true,
        required: true
    },
    photo: {
        type: String,
        required: true
    },
    kapasitas: {
        type: Number,
        required: true
    },
    status: {
        type: String,
        enum: ["Tersedia", "Tidak Tersedia"],
        default: "Tersedia"
    }
}, { timestamps: true });

const BookingModel = mongoose.model('Booking', BookingSchema);
const WorkshopModel = mongoose.model('Workshop', WorkshopSchema);

// Middleware untuk update status workshop setelah booking
BookingSchema.post('save', async function() {
    const workshop = await WorkshopModel.findById(this.workshop);
    const confirmedBookings = await BookingModel.countDocuments({ workshop: workshop._id, returned: false });

    if (confirmedBookings >= workshop.kapasitas) {
        workshop.status = "Tidak Tersedia";
    } else {
        workshop.status = "Tersedia";
    }

    await workshop.save();
});

module.exports = { BookingModel, WorkshopModel };
