const path = require('path');
const mongoose = require('mongoose');
const express = require('express');
const isAuthenticated = require('../middleware/authenticateToken');
const { AsramaBookingModel, AsramaModel } = require('../models/Asrama'); 
const UsersModel = require('../models/Users');

const router = express.Router();

//#ONLY ADMIN

// Get all asramas
router.get('/getAllAsrama', isAuthenticated, async (req, res) => {
    try {
        const asramas = await AsramaModel.find();
        res.json(asramas);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Create Asrama
router.post('/createAsrama', isAuthenticated, async (req, res) => {
    const { name, capacity } = req.body;

    if (!req.files || !req.files.file) {
        return res.status(400).json({ msg: 'No files were uploaded.' });
    }

    const file = req.files.file;
    const allowedType = ['.png', '.jpg', '.jpeg'];
    const fileSizeLimit = 5000000; // 5MB

    try {
        validateFile(file, allowedType, fileSizeLimit);

        const { fileName, url } = generateFileNameAndUrl(file, req);

        await moveFile(file, fileName);

        const newAsrama = new AsramaModel({
            name,
            photo: fileName,
            url,
            capacity,
        });

        await newAsrama.save();
        res.status(201).json({ msg: 'Meeting room created successfully', room: newAsrama });

    } catch (error) {
        console.error(error.message);
        res.status(500).json({ msg: 'Server error', error: error.message });
    }
});


function validateFile(file, allowedType, fileSizeLimit) {
    const ext = path.extname(file.name).toLowerCase();
    if (!allowedType.includes(ext)) {
        throw new Error('Invalid image format!');
    }
    if (file.size > fileSizeLimit) {
        throw new Error('Image must be less than 5MB');
    }
}

function generateFileNameAndUrl(file, req) {
    const ext = path.extname(file.name).toLowerCase();
    const fileName = file.md5 + ext;
    const url = `https://api-sarpras.bbpvpsemarang.com/images/${fileName}`;
    return { fileName, url };
}

function moveFile(file, fileName) {
    return new Promise((resolve, reject) => {
        file.mv(`./public/images/${fileName}`, (err) => {
            if (err) {
                return reject(new Error('File upload failed.'));
            }
            resolve();
        });
    });
}


// Update Asrama
router.patch('/updateAsrama/:id', isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { name, capacity } = req.body;
      const asrama = await AsramaModel.findByIdAndUpdate(id, { name, capacity }, { new: true });
      res.json(asrama);
    } catch (error) {
      res.status(500).json({ message: 'Error updating asrama' });
    }
  });
  
  // Delete Asrama
  router.delete('/deleteAsrama/:id', isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      await AsramaModel.findByIdAndDelete(id);
      res.json({ message: 'Asrama deleted successfully' });
    } catch (error) {
      res.status(500).json({ message: 'Error deleting asrama' });
    }
  });
  

//#ONLY USER

//Get all bookings by Admin
router.get('/getAllBookingUser', isAuthenticated, async (req, res) => {
    try {
        const bookings = await AsramaBookingModel.find().populate('asrama'); 
        res.json(bookings);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

//Get all bookings by User
router.get('/getAllBookings', isAuthenticated, async (req, res) => {
    try {
        const { email } = req.user;
        const user = await UsersModel.findOne({ email });

        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }
        const bookings = await AsramaBookingModel.find({ user_id: user._id }).populate('asrama'); 
        res.json(bookings);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Get all bookings
router.post('/addbooking', isAuthenticated, async (req, res) => {
    try {
        const { email } = req.user;
        const user = await UsersModel.findOne({ email });

        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }

        const { borrower, borrowDate, returnedDate, notes, asrama } = req.body;

        // Validate required fields
        if (!borrower || !borrowDate || !returnedDate || !asrama) {
            return res.status(400).json({ message: 'All required fields must be provided.' });
        }

        // Convert borrowDate and returnedDate to Date objects
        const borrowDateObj = new Date(borrowDate);
        const returnedDateObj = new Date(returnedDate);

        // Check if there is any overlapping booking that has been approved
        const overlappingBooking = await AsramaBookingModel.findOne({
            asrama: new mongoose.Types.ObjectId(asrama),
            status: "disetujui",
            $or: [
                { borrowDate: { $lt: returnedDateObj }, returnedDate: { $gt: borrowDateObj } }, // Overlap between two dates
                { borrowDate: { $eq: borrowDateObj } }, // Exact same borrow date
                { returnedDate: { $eq: returnedDateObj } } // Exact same return date
            ]
        });

        if (overlappingBooking) {
            return res.status(400).json({ message: 'The selected date range overlaps with an existing approved booking.' });
            
        }

        // Create the new booking
        const newBooking = new AsramaBookingModel({
            user_id: user._id,
            borrower,
            borrowDate: borrowDateObj,
            returnedDate: returnedDateObj,
            notes,
            asrama: new mongoose.Types.ObjectId(asrama),
            status: "menunggu"
        });

        // Save the new booking to the database
        const savedBooking = await newBooking.save();

        res.status(201).json({ message: 'Booking created successfully.', booking: savedBooking });

    } catch (error) {
        console.error(error.message);
        res.status(500).json({ msg: 'Server error', error: error.message });
    }
});



// Update booking by ID
router.put('/updateBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const booking = await AsramaBookingModel.findById(id).populate('asrama');
        if (!booking) {
            return res.status(404).json({ error: 'Booking not found' });
        }

        if (booking.status !== "menunggu") {
            return res.status(400).json({ error: 'Booking is not in a pending state' });
        }

        booking.status = "disetujui";
        booking.ketersediaan = "Tidak Tersedia";
        await booking.save();

        res.status(200).json(booking);
    } catch (error) {
        console.error('Error approving booking:', error);
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Reject booking by ID
router.put('/rejectBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const booking = await AsramaBookingModel.findById(id).populate('asrama');
        if (!booking) {
            return res.status(404).json({ error: 'Booking not found' });
        }
        if (booking.status !== "menunggu") {
            return res.status(400).json({ error: 'Booking is not in a pending state' });
        }

        booking.status = "ditolak";
        
        await booking.save();

        res.status(200).json(booking);
    } catch (error) {
        console.error('Error rejecting booking:', error);
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

//finish booking by ID
router.put('/finishBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const booking = await AsramaBookingModel.findById(id).populate('asrama');
        if (!booking) {
            return res.status(404).json({ error: 'Booking not found' });
        }

        if (booking.status !== "disetujui") {
            return res.status(400).json({ error: 'Booking is not in a pending state' });
        }

        booking.status = "selesai";
        booking.ketersediaan = "Tersedia";
        await booking.save();

        res.status(200).json(booking);
    } catch (error) {
        console.error('Error approving booking:', error);
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Delete booking by ID
router.delete('/deleteBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const deletedBooking = await BookingModel.findByIdAndDelete(id);

        if (!deletedBooking) {
            return res.status(404).json({ error: 'Booking not found' });
        }

        // Update the status of the `asrama` after deleting the booking
        const asramaDoc = await AsramaModel.findById(deletedBooking.asrama);
        if (asramaDoc) {
            const confirmedBookings = await BookingModel.countDocuments({
                asrama: asramaDoc._id,
                returned: false
            });

            asramaDoc.status = (confirmedBookings >= asramaDoc.kapasitas) ? "Tidak Tersedia" : "Tersedia";
            await asramaDoc.save();
        }

        res.json({ message: 'Booking deleted successfully' });
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

router.get('/getTotalBookingUnavailable', isAuthenticated, async (req, res) => {
    try {
        const totalUnavailableBookings = await AsramaBookingModel.aggregate([
            {
                $match: { ketersediaan: "Tidak Tersedia" } // Filter based on availability
            },
            {
                $group: {
                    _id: null,
                    total: { $sum: 1 } // Count documents with unavailable status
                }
            }
        ]);

        const total = await AsramaModel.aggregate([
            {
                $group: {
                    _id: null,
                    total: { $sum: 1 } // Count all documents in the collection
                }
            }
        ]);

        const totalBook= totalUnavailableBookings.length > 0 ? totalUnavailableBookings[0].total : 0;

        const totalAsrama = total.length > 0 ? total[0].total : 0;

        res.json({
           total : {
            totalBook: totalBook,
            totalAsrama: totalAsrama
           }
        });
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});





module.exports = router;
