const express = require('express');
const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const UsersModel = require('../models/Users');
const { JWT_SECRET } = require('../config/config');

const router = express.Router();

router.post('/register', async (req, res) => {
    try {
        const { name, email, password } = req.body;

        if (password.length < 8 || !/[A-Z]/.test(password) || !/[a-z]/.test(password) || !/[0-9]/.test(password)) {
            return res.status(400).json({
                error: 'Password must be at least 8 characters long, contain at least one uppercase letter, one lowercase letter, and one digit.'
            });
        }

        const passwordHash = await bcrypt.hash(password, await bcrypt.genSalt());

        await UsersModel.create({ name, email, password: passwordHash });

        res.json({ status: 'Success', message: 'User registered successfully' });
    } catch (error) {
        if (error.code === 11000) {
            return res.status(400).json({ error: 'Email is already registered' });
        }
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

router.post('/login', async (req, res) => {
    try {
        const { email, password } = req.body;
        const user = await UsersModel.findOne({ email });

        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }

        if (await bcrypt.compare(password, user.password)) {
            const token = jwt.sign({ email: user.email, role: user.role }, JWT_SECRET, { expiresIn: '7d' }); // Token expires in 7 days

            // Set cookie with token
            res.cookie('token', token, {
                httpOnly: true, // Prevents JavaScript access
                secure: true, // Ensures cookie is sent over HTTPS
                sameSite: 'strict', // Protects against CSRF attacks
                maxAge: 7 * 24 * 60 * 60 * 1000 // 7 days in milliseconds
            });

            res.json({ Status: 'Success', role: user.role, displayName: user.name , token: token});
        } else {
            res.status(401).json({ message: 'Password is incorrect' });
        }
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});


router.post('/logout', (req, res) => {
    res.clearCookie('token').sendStatus(200);
});

module.exports = router;
