const express = require('express');
const mongoose = require('mongoose');
const path = require('path');
const isAuthenticated = require('../middleware/authenticateToken');
const { KendaraanModel , KendaraanBookingModel} = require('../models/Kendaraan'); 
const UsersModel = require('../models/Users');

const router = express.Router();

//#ONLY ADMIN

// Get All Data
router.get('/getAllKendaraan', isAuthenticated, async (req, res) => {
    try {
        const meetingroom = await KendaraanModel.find();
        res.json(meetingroom);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Create 
router.post('/createKendaraan', isAuthenticated, async (req, res) => {
    const { nopolisi, name, norangka, nomesin, nobpkb } = req.body;

    if (!req.files || !req.files.file) {
        return res.status(400).json({ msg: 'No files were uploaded.' });
    }

    const file = req.files.file;
    const allowedType = ['.png', '.jpg', '.jpeg'];
    const fileSizeLimit = 5000000; // 5MB

    try {
        validateFile(file, allowedType, fileSizeLimit);

        const { fileName, url } = generateFileNameAndUrl(file, req);

        await moveFile(file, fileName);

        const newKendaraan = new KendaraanModel({
            photo: fileName,
            url,
            nopolisi,
            name,
            norangka,
            nomesin,
            nobpkb
        });

        await newKendaraan.save();
        res.status(201).json({ msg: 'Kendaraan created successfully', room: newKendaraan });
    } catch (error) {
        console.error(error.message);
        res.status(500).json({ msg: 'Server error', error: error.message });
    }
});

router.patch('/updateKendaraan/:id', isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { name, nopolisi, norangka, nomesin, nobpkb } = req.body;
      const kendaraan = await KendaraanModel.findByIdAndUpdate(id, { name, nopolisi, norangka, nomesin, nobpkb }, { new: true });
      res.json(kendaraan);
    } catch (error) {
      res.status(500).json({ message: 'Error updating asrama' });
    }
  });
  
  router.delete('/deleteKendaraan/:id', isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      await KendaraanModel.findByIdAndDelete(id);
      res.json({ message: 'Kendaraan deleted successfully' });
    } catch (error) {
      res.status(500).json({ message: 'Error deleting kendaraan' });
    }
  });
  

function validateFile(file, allowedType, fileSizeLimit) {
    const ext = path.extname(file.name).toLowerCase();
    if (!allowedType.includes(ext)) {
        throw new Error('Invalid image format!');
    }
    if (file.size > fileSizeLimit) {
        throw new Error('Image must be less than 5MB');
    }
}

function generateFileNameAndUrl(file, req) {
    const ext = path.extname(file.name).toLowerCase();
    const fileName = file.md5 + ext;
    const url = `${req.protocol}://${req.get("host")}/images/${fileName}`;
    return { fileName, url };
}

function moveFile(file, fileName) {
    return new Promise((resolve, reject) => {
        file.mv(`./public/images/${fileName}`, (err) => {
            if (err) {
                return reject(new Error('File upload failed.'));
            }
            resolve();
        });
    });
}

//Only User
//Get All Booking by Admin
router.get('/getAllBookingUser', isAuthenticated, async (req, res) => {
    try {
        const bookings = await KendaraanBookingModel.find().populate('kendaraan'); 
        res.json(bookings);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

//Get All Booking By User
router.get('/getAllBookings', isAuthenticated, async (req, res) => {
    try {
        const { email } = req.user;
        const user = await UsersModel.findOne({ email });

        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }
        const bookings = await KendaraanBookingModel.find({ user_id: user._id }).populate('kendaraan'); 
        res.json(bookings);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

router.post('/addbooking', isAuthenticated, async (req, res) => {
    try {
        const { email } = req.user;
    const user = await UsersModel.findOne({ email });

    if (!user) {
        return res.status(404).json({ message: 'User not found' });
    }

    const { borrower, borrowDate, returnDate, startTime, endTime, keperluan, notes, kendaraan } = req.body;

     // Convert times to Date objects for comparison
     const start = new Date(`${borrowDate}T${startTime}`);
     const end = new Date(`${borrowDate}T${endTime}`);

      // Convert borrowDate and returnedDate to Date objects
      const borrowDateObj = new Date(borrowDate);
      const returnedDateObj = new Date(returnDate);

      // Check if there is any overlapping booking that has been approved
      const overlappingBooking = await KendaraanBookingModel.findOne({
        kendaraan: new mongoose.Types.ObjectId(kendaraan),
          status: "disetujui",
          $or: [
              { borrowDate: { $lt: returnedDateObj }, returnDate: { $gt: borrowDateObj } }, // Overlap between two dates
              { borrowDate: { $eq: borrowDateObj } }, // Exact same borrow date
              { returnDate: { $eq: returnedDateObj } } // Exact same return date
          ]
      });

      if (overlappingBooking) {
          return res.status(400).json({ message: 'The selected date range overlaps with an existing approved booking.' });
          
      }

    // Validate required fields
    if ( !borrower || !borrowDate || !returnDate || !startTime || !endTime || !kendaraan) {
        return res.status(400).json({ message: 'All required fields must be provided.' });
    }

    // Create the new booking
    const newBooking = new KendaraanBookingModel({
        user_id: user._id,
        borrower,
        borrowDate: borrowDateObj,
        returnDate: returnedDateObj,
        startTime: start.toISOString(),
        endTime: end.toISOString(),
        keperluan,
        notes,
        kendaraan: new mongoose.Types.ObjectId(kendaraan),
        status: "menunggu"
    });

    // Save the new booking to the database
    const savedBooking = await newBooking.save();

    res.status(201).json({ message: 'Booking created successfully.', booking: savedBooking });

    } catch (error) {
        console.error(error.message);
        res.status(500).json({ msg: 'Server error', error: error.message });
    }
})

// Update booking by ID
router.put('/updateBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const booking = await KendaraanBookingModel.findById(id).populate('kendaraan');
        if (!booking) {
            return res.status(404).json({ error: 'Booking not found' });
        }

        if (booking.status !== "menunggu") {
            return res.status(400).json({ error: 'Booking is not in a pending state' });
        }

        booking.status = "disetujui";
        booking.ketersediaan = "Tidak Tersedia";
        await booking.save();

        res.status(200).json(booking);
    } catch (error) {
        console.error('Error approving booking:', error);
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Reject booking by ID
router.put('/rejectBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const booking = await KendaraanBookingModel.findById(id).populate('kendaraan');
        if (!booking) {
            return res.status(404).json({ error: 'Booking not found' });
        }
        if (booking.status !== "menunggu") {
            return res.status(400).json({ error: 'Booking is not in a pending state' });
        }

        booking.status = "ditolak";
        
        await booking.save();

        res.status(200).json(booking);
    } catch (error) {
        console.error('Error rejecting booking:', error);
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

//finish booking by ID
router.put('/finishBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const booking = await KendaraanBookingModel.findById(id).populate('kendaraan');
        if (!booking) {
            return res.status(404).json({ error: 'Booking not found' });
        }

        if (booking.status !== "disetujui") {
            return res.status(400).json({ error: 'Booking is not in a pending state' });
        }

        booking.status = "selesai";
        booking.ketersediaan = "Tersedia";
        await booking.save();

        res.status(200).json(booking);
    } catch (error) {
        console.error('Error approving booking:', error);
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

router.get('/getTotalBookingUnavailable', isAuthenticated, async (req, res) => {
    try {
        const totalUnavailableBookings = await KendaraanBookingModel.aggregate([
            {
                $match: { ketersediaan: "Tidak Tersedia" } // Filter based on availability
            },
            {
                $group: {
                    _id: null,
                    total: { $sum: 1 } // Count documents with unavailable status
                }
            }
        ]);

        const total = await KendaraanModel.aggregate([
            {
                $group: {
                    _id: null,
                    total: { $sum: 1 } // Count all documents in the collection
                }
            }
        ]);

        const totalBook = totalUnavailableBookings.length > 0 ? totalUnavailableBookings[0].total : 0;
        const totalKendaraan = total.length > 0 ? total[0].total : 0;

        res.json({
            total:{
                totalBook : totalBook,
                totalKendaraan : totalKendaraan
            }
        });
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});


module.exports = router