const express = require('express');
const mongoose = require('mongoose');
const path = require('path');
const isAuthenticated = require('../middleware/authenticateToken');
const { MeetingRoomModel , MeetingBookingModel} = require('../models/RuangRapat'); 
const UsersModel = require('../models/Users');


const router = express.Router();

//#ONLY ADMIN

// Get All Data
router.get('/getAllMeetingRoom', isAuthenticated, async (req, res) => {
    try {
        const meetingroom = await MeetingRoomModel.find();
        res.json(meetingroom);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Create 
router.post('/createMeetingRoom', isAuthenticated, async (req, res) => {
    const { name, capacity } = req.body;

    if (!req.files || !req.files.file) {
        return res.status(400).json({ msg: 'No files were uploaded.' });
    }

    const file = req.files.file;
    const allowedType = ['.png', '.jpg', '.jpeg'];
    const fileSizeLimit = 5000000; // 5MB

    try {
        validateFile(file, allowedType, fileSizeLimit);

        const { fileName, url } = generateFileNameAndUrl(file, req);

        await moveFile(file, fileName);

        const newMeetingRoom = new MeetingRoomModel({
            name,
            photo: fileName,
            url,
            capacity,
        });

        await newMeetingRoom.save();
        res.status(201).json({ msg: 'Meeting room created successfully', room: newMeetingRoom });

    } catch (error) {
        console.error(error.message);
        res.status(500).json({ msg: 'Server error', error: error.message });
    }
});

// Update Asrama
router.patch('/updateMeetingRoom/:id', isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { name, capacity } = req.body;
      const meetingRoom = await MeetingRoomModel.findByIdAndUpdate(id, { name, capacity }, { new: true });
      res.json(meetingRoom);
    } catch (error) {
      res.status(500).json({ message: 'Error updating Meeting Room' });
    }
  });
  
  // Delete Asrama
  router.delete('/deleteMeetingRoom/:id', isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      await MeetingRoomModel.findByIdAndDelete(id);
      res.json({ message: 'Meeting room deleted successfully' });
    } catch (error) {
      res.status(500).json({ message: 'Error deleting Meeting Room' });
    }
  });

function validateFile(file, allowedType, fileSizeLimit) {
    const ext = path.extname(file.name).toLowerCase();
    if (!allowedType.includes(ext)) {
        throw new Error('Invalid image format!');
    }
    if (file.size > fileSizeLimit) {
        throw new Error('Image must be less than 5MB');
    }
}

function generateFileNameAndUrl(file, req) {
    const ext = path.extname(file.name).toLowerCase();
    const fileName = file.md5 + ext;
    const url = `${req.protocol}://${req.get("host")}/images/${fileName}`;
    return { fileName, url };
}

function moveFile(file, fileName) {
    return new Promise((resolve, reject) => {
        file.mv(`./public/images/${fileName}`, (err) => {
            if (err) {
                return reject(new Error('File upload failed.'));
            }
            resolve();
        });
    });
}


// Update 
router.patch('/updateMeetingRoom/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;
        const { name, capacity } = req.body;
        const meetingroom = await MeetingRoomModel.findByIdAndUpdate(id, { name, capacity }, { new: true });
        res.json(meetingroom);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Delete
router.delete('/deleteMeetingRoom/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;
        const meetingroom = await MeetingRoomModel.findByIdAndDelete(id);
        res.json(meetingroom);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

//#ONLY USER
router.get('/getAllBookingUser', isAuthenticated, async (req, res) => {
    try {
        const bookings = await MeetingBookingModel.find().populate('meetingRoom'); 
        res.json(bookings);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Get all bookings
router.get('/getAllBookings', isAuthenticated, async (req, res) => {
    try {
        const { email } = req.user;
        const user = await UsersModel.findOne({ email });

        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }

        const bookings = await MeetingBookingModel.find({ user_id: user._id }).populate('meetingRoom'); 
        res.json(bookings);
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Create a new booking
router.post('/addbooking', isAuthenticated, async (req, res) => {
    try {
        const { email } = req.user;
        const user = await UsersModel.findOne({ email });

        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }

        const { borrower, borrowDate, startTime, endTime, notes, meetingRoom } = req.body;

        // Validate required fields
        if (!borrower || !borrowDate || !startTime || !endTime || !meetingRoom) {
            return res.status(400).json({ message: 'All required fields must be provided.' });
        }

        // Convert times to Date objects for comparison
        const start = new Date(`${borrowDate}T${startTime}`);
        const end = new Date(`${borrowDate}T${endTime}`);

        // Check if start time is before end time
        if (start >= end) {
            return res.status(400).json({ message: 'Start time must be before end time.' });
        }

        // Check if there is any overlapping booking that has been approved
        const overlappingBooking = await MeetingBookingModel.findOne({
            meetingRoom: new mongoose.Types.ObjectId(meetingRoom),
            status: "disetujui",
            borrowDate,
            $or: [
                // Booking times overlap with existing bookings
                { startTime: { $lt: end.toISOString() }, endTime: { $gt: start.toISOString() } }
            ]
        });

        if (overlappingBooking) {
            return res.status(400).json({ message: 'The selected time range overlaps with an existing approved booking.' });
        }

        // Create the new booking
        const newBooking = new MeetingBookingModel({
            user_id: user._id,
            borrower,
            borrowDate,
            startTime: start.toISOString(),
            endTime: end.toISOString(),
            notes,
            meetingRoom: new mongoose.Types.ObjectId(meetingRoom),
            status: "menunggu"
        });

        // Save the new booking to the database
        const savedBooking = await newBooking.save();

        res.status(201).json({ message: 'Booking created successfully.', booking: savedBooking });

    } catch (error) {
        console.error(error.message);
        res.status(500).json({ msg: 'Server error', error: error.message });
    }
});

// Update booking by ID
router.put('/updateBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const booking = await MeetingBookingModel.findById(id).populate('meetingRoom');
        if (!booking) {
            return res.status(404).json({ error: 'Booking not found' });
        }

        if (booking.status !== "menunggu") {
            return res.status(400).json({ error: 'Booking is not in a pending state' });
        }

        booking.status = "disetujui";
        booking.ketersediaan = "Tidak Tersedia";
        await booking.save();

        res.status(200).json(booking);
    } catch (error) {
        console.error('Error approving booking:', error);
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

// Reject booking by ID
router.put('/rejectBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const booking = await MeetingBookingModel.findById(id).populate('meetingRoom');
        if (!booking) {
            return res.status(404).json({ error: 'Booking not found' });
        }
        if (booking.status !== "menunggu") {
            return res.status(400).json({ error: 'Booking is not in a pending state' });
        }

        booking.status = "ditolak";
        
        await booking.save();

        res.status(200).json(booking);
    } catch (error) {
        console.error('Error rejecting booking:', error);
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

//finish booking by ID
router.put('/finishBooking/:id', isAuthenticated, async (req, res) => {
    try {
        const { id } = req.params;

        const booking = await MeetingBookingModel.findById(id).populate('meetingRoom');
        if (!booking) {
            return res.status(404).json({ error: 'Booking not found' });
        }

        if (booking.status !== "disetujui") {
            return res.status(400).json({ error: 'Booking is not in a pending state' });
        }

        booking.status = "selesai";
        booking.ketersediaan = "Tersedia";
        await booking.save();

        res.status(200).json(booking);
    } catch (error) {
        console.error('Error approving booking:', error);
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

router.get('/getTotalBookingUnavailable', isAuthenticated, async (req, res) => {
    try {
        const totalUnavailableBookings = await MeetingBookingModel.aggregate([
            {
                $match: { ketersediaan: "Tidak Tersedia" } // Filter based on availability
            },
            {
                $group: {
                    _id: null,
                    total: { $sum: 1 } // Count documents with unavailable status
                }
            }
        ]);

        const total = await MeetingRoomModel.aggregate([
            {
                $group: {
                    _id: null,
                    total: { $sum: 1 } // Count all documents in the collection
                }
            }
        ]);

        const totalBook = totalUnavailableBookings.length > 0 ? totalUnavailableBookings[0].total : 0;
        const totalMeetingRoom = total.length > 0 ? total[0].total : 0;

        res.json({
            total:{
                totalBook : totalBook,
                totalMeetingRoom : totalMeetingRoom
            }
        });
    } catch (error) {
        res.status(500).json({ error: 'Internal Server Error' });
    }
});

module.exports = router;